﻿using Coscine.Database.DataModel;
using Coscine.Database.ReturnObjects;
using Coscine.Database.Util;

using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;

namespace Coscine.Database.Models
{
    public class ResourceModel : DatabaseModel<Resource>
    {
        public Resource StoreFromObject(ResourceObject resourceObject)
        {
            if (!resourceObject.Disciplines.Any() || resourceObject.ResourceTypeOption == null)
            {
                throw new ArgumentException("Discipline and ResourceTypeOption are necessary!");
            }

            Resource resource = new Resource()
            {
                DisplayName = resourceObject.DisplayName,
                ResourceName = resourceObject.ResourceName,
                Description = resourceObject.Description,
                Keywords = resourceObject.Keywords,
                UsageRights = resourceObject.UsageRights,
                TypeId = resourceObject.Type.Id,
                VisibilityId = resourceObject.Visibility.Id,
                ApplicationProfile = resourceObject.ApplicationProfile,
                FixedValues = resourceObject.FixedValues != null ? resourceObject.FixedValues.ToString() : "{}",
                Creator = resourceObject.Creator
            };
            if (resourceObject.License != null)
            {
                resource.LicenseId = resourceObject.License.Id;
            }
            Insert(resource);

            resource.Type = new ResourceTypeModel().GetById(resourceObject.Type.Id);
            try
            {
                SetDisciplines(resource, resourceObject.Disciplines);
                SetResourceTypeObject(resource, resourceObject.ResourceTypeOption);
            }
            catch (Exception e)
            {
                Delete(resource);
                throw e;
            }

            return resource;
        }

        public int DeleteResource(Resource resource)
        {
            ProjectResourceModel projectResourceModel = new ProjectResourceModel();
            foreach (var projectResource in projectResourceModel.GetAllWhere((projectResource) => projectResource.ResourceId == resource.Id))
            {
                projectResourceModel.Delete(projectResource);
            }

            ResourceDisciplineModel resourceDisciplineModel = new ResourceDisciplineModel();
            foreach (var resourceDiscipline in resourceDisciplineModel.GetAllWhere((resourceDicipline) => resourceDicipline.ResourceId == resource.Id))
            {
                resourceDisciplineModel.Delete(resourceDiscipline);
            }

            DeleteResourceTypeObject(resource);

            return Delete(resource);
        }

        public List<OrganizationCountObject> GetRDSBucketCountByOrganization()
        {
            var resourceListByOrganization = GetRDSResourceListByOrganization();
            var list = new List<OrganizationCountObject>();
            foreach (OrganizationResourceListObject rlo in resourceListByOrganization)
            {
                list.Add(new OrganizationCountObject(rlo.Url, rlo.Resources.Count));
            }
            return list;
        }

        public List<OrganizationResourceListObject> GetRDSResourceListByOrganization()
        {
            return DatabaseConnection.ConnectToDatabase((db) =>
            {
                return (from r in db.Resources
                        join pr in db.ProjectResources on r.Id equals pr.ResourceId into joinedPr
                        from jpr in joinedPr.DefaultIfEmpty()
                        join p in db.Projects on jpr.ProjectId equals p.Id into joinedP
                        from jp in joinedP.DefaultIfEmpty()
                        join pi in db.ProjectInstitutes on jp.Id equals pi.ProjectId into joinedPi
                        from jpi in joinedPi.DefaultIfEmpty()
                        join rt in db.ResourceTypes on r.TypeId equals rt.Id into joinedRt
                        from jrt in joinedRt.DefaultIfEmpty()

                        where jp.Deleted == false &&
                        jrt.DisplayName == "rds"
                        group r by jpi.OrganizationUrl into g
                        select new OrganizationResourceListObject(g.Key, g.ToList())).ToList();
            });
        }

        private void SetResourceTypeObject(Resource resource, JObject resourceTypeOption)
        {
            if (resource.Type.DisplayName == "rds")
            {
                RDSResourceTypeObject rdsResourceTypeObject = resourceTypeOption.ToObject<RDSResourceTypeObject>();
                RDSResourceTypeModel rdsResourceTypeModel = new RDSResourceTypeModel();
                if (resource.ResourceTypeOptionId != null)
                {
                    RdsresourceType rdsResourceType = rdsResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);
                    rdsResourceTypeModel.Update(rdsResourceType);
                }
                else
                {
                    RdsresourceType rdsResourceType = new RdsresourceType()
                    {
                        BucketName = resource.Id.ToString(),
                        Size = rdsResourceTypeObject.Size,
                    };
                    rdsResourceTypeModel.Insert(rdsResourceType);
                    resource.ResourceTypeOptionId = rdsResourceType.Id;
                    Update(resource);
                }
            }
            else if (resource.Type.DisplayName == "s3")
            {
                S3ResourceTypeObject s3ResourceTypeObject = resourceTypeOption.ToObject<S3ResourceTypeObject>();
                S3ResourceTypeModel s3ResourceTypeModel = new S3ResourceTypeModel();
                if (resource.ResourceTypeOptionId != null)
                {
                    S3resourceType s3ResourceType = s3ResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);

                    s3ResourceType.BucketName = s3ResourceTypeObject.BucketName;
                    s3ResourceType.AccessKey = s3ResourceTypeObject.AccessKey ?? s3ResourceType.AccessKey;
                    s3ResourceType.SecretKey = s3ResourceTypeObject.SecretKey ?? s3ResourceType.SecretKey;

                    s3ResourceTypeModel.Update(s3ResourceType);
                }
                else
                {
                    S3resourceType s3ResourceType = new S3resourceType()
                    {
                        BucketName = s3ResourceTypeObject.BucketName,
                        AccessKey = s3ResourceTypeObject.AccessKey,
                        SecretKey = s3ResourceTypeObject.SecretKey
                    };
                    s3ResourceTypeModel.Insert(s3ResourceType);
                    resource.ResourceTypeOptionId = s3ResourceType.Id;
                    Update(resource);
                }
            }
            else if (resource.Type.DisplayName == "gitlab")
            {
                GitlabResourceTypeObject gitlabResourceTypeObject = resourceTypeOption.ToObject<GitlabResourceTypeObject>();
                GitlabResourceTypeModel gitlabResourceTypeModel = new GitlabResourceTypeModel();
                if (resource.ResourceTypeOptionId != null)
                {
                    GitlabResourceType gitlabResourceType = gitlabResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);

                    gitlabResourceType.RepositoryNumber = gitlabResourceTypeObject.RepositoryNumber;
                    gitlabResourceType.RepositoryUrl = gitlabResourceTypeObject.RepositoryUrl;
                    gitlabResourceType.Token = gitlabResourceTypeObject.Token ?? gitlabResourceType.Token;

                    gitlabResourceTypeModel.Update(gitlabResourceType);
                }
                else
                {
                    GitlabResourceType gitlabResourceType = new GitlabResourceType()
                    {
                        RepositoryNumber = gitlabResourceTypeObject.RepositoryNumber,
                        RepositoryUrl = gitlabResourceTypeObject.RepositoryUrl,
                        Token = gitlabResourceTypeObject.Token
                    };
                    gitlabResourceTypeModel.Insert(gitlabResourceType);
                    resource.ResourceTypeOptionId = gitlabResourceType.Id;
                    Update(resource);
                }
            }
            else if (resource.Type.DisplayName == "linked")
            {
                LinkedResourceTypeModel linkedResourceTypeModel = new LinkedResourceTypeModel();
                if (resource.ResourceTypeOptionId == null)
                {
                    LinkedResourceType linkedResourceType = new LinkedResourceType();
                    linkedResourceTypeModel.Insert(linkedResourceType);
                    resource.ResourceTypeOptionId = linkedResourceType.Id;
                    Update(resource);
                }
            }
            else
            {
                throw new ArgumentException("Not supported resource type!");
            }
        }

        public Dictionary<string, string> GetResourceTypeOptions(Guid id)
        {
            var resource = GetById(id);
            var resourceType = new ResourceTypeModel().GetById((Guid)resource.TypeId);
            var resourceTypeOptionId = (Guid)resource.ResourceTypeOptionId;
            if (resourceType.DisplayName == "gitlab")
            {
                return new GitlabResourceTypeModel().GetResourceTypeOptions(resourceTypeOptionId);
            }
            else if (resourceType.DisplayName == "rds")
            {
                return new RDSResourceTypeModel().GetResourceTypeOptions(resourceTypeOptionId);
            }
            else if (resourceType.DisplayName == "s3")
            {
                return new S3ResourceTypeModel().GetResourceTypeOptions(resourceTypeOptionId);
            }
            else if (resourceType.DisplayName == "linked")
            {
                return new LinkedResourceTypeModel().GetResourceTypeOptions(resourceTypeOptionId);
            }
            else
            {
                return new Dictionary<string, string>();
            }
        }

        private void SetDisciplines(Resource resource, IEnumerable<DisciplineObject> disciplines)
        {
            ResourceDisciplineModel resourceDisciplineModel = new ResourceDisciplineModel();
            foreach (var oldDiscipline in resourceDisciplineModel.GetAllWhere((resourceDiscipline) => resourceDiscipline.ResourceId == resource.Id))
            {
                resourceDisciplineModel.Delete(oldDiscipline);
            }
            foreach (var discipline in disciplines)
            {
                ResourceDiscipline resourceDiscipline = new ResourceDiscipline()
                {
                    DisciplineId = discipline.Id,
                    ResourceId = resource.Id
                };
                resourceDisciplineModel.Insert(resourceDiscipline);
            }
        }

        public bool HasAccess(User user, Resource resource, params string[] allowedAccess)
        {
            var projectId = new ProjectResourceModel().GetProjectForResource(resource.Id);
            IEnumerable<string> allowedAccessLabels = allowedAccess.Select(x => x.ToLower().Trim()).ToList();
            return DatabaseConnection.ConnectToDatabase((db) => (from relation in db.ProjectRoles
                                                                 where relation.ProjectId == projectId
                                                                     && relation.User.Id == user.Id
                                                                     && allowedAccessLabels.Contains(relation.Role.DisplayName.ToLower())
                                                                 select relation).Any());
        }

        public int UpdateByObject(Resource resource, ResourceObject resourceObject)
        {
            if (resourceObject.Disciplines.Count() == 0 || resourceObject.ResourceTypeOption == null)
            {
                throw new ArgumentException("Discipline and ResourceTypeOption are necessary!");
            }

            if (resource.TypeId != resourceObject.Type.Id)
            {
                DeleteResourceTypeObject(resource);
            }

            resource.DisplayName = resourceObject.DisplayName;
            resource.ResourceName = resourceObject.ResourceName;
            resource.Description = resourceObject.Description;
            resource.Keywords = resourceObject.Keywords;
            resource.UsageRights = resourceObject.UsageRights;
            resource.TypeId = resourceObject.Type.Id;
            resource.Type = new ResourceTypeModel().GetById(resourceObject.Type.Id);
            resource.VisibilityId = resourceObject.Visibility.Id;
            if (resourceObject.License != null)
            {
                resource.LicenseId = resourceObject.License.Id;
            }

            // the application profile can not be altered after creation
            // resource.ApplicationProfile = resourceObject.ApplicationProfile;

            resource.FixedValues = resourceObject.FixedValues != null ? resourceObject.FixedValues.ToString() : "{}";

            // the resource creator can not be altered after creation
            // resource.Creator = resourceObject.Creator;

            SetDisciplines(resource, resourceObject.Disciplines);
            SetResourceTypeObject(resource, resourceObject.ResourceTypeOption);

            return Update(resource);
        }

        // TODO: Find out why resource.Type is not set
        public void SetType(Resource resource)
        {
            if (resource.Type == null)
            {
                ResourceTypeModel resourceTypeModel = new ResourceTypeModel();
                resource.Type = resourceTypeModel.GetById(resource.TypeId);
            }
        }

        public void DeleteResourceTypeObject(Resource resource)
        {
            SetType(resource);
            if (resource.Type.DisplayName == "rds" && resource.ResourceTypeOptionId != null)
            {
                RDSResourceTypeModel rdsResourceTypeModel = new RDSResourceTypeModel();
                rdsResourceTypeModel.Delete(rdsResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value));
            }
            else if (resource.Type.DisplayName == "s3" && resource.ResourceTypeOptionId != null)
            {
                S3ResourceTypeModel s3ResourceTypeModel = new S3ResourceTypeModel();
                s3ResourceTypeModel.Delete(s3ResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value));
            }
            else if (resource.Type.DisplayName == "gitlab" && resource.ResourceTypeOptionId != null)
            {
                GitlabResourceTypeModel gitlabResourceTypeModel = new GitlabResourceTypeModel();
                gitlabResourceTypeModel.Delete(gitlabResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value));
            }
        }

        public ResourceObject CreateReturnObjectFromDatabaseObject(Resource resource)
        {
            SetType(resource);

            DisciplineModel disciplineModel = new DisciplineModel();
            var disciplines = disciplineModel.GetAllWhere((discipline) =>
                            (from relation in discipline.ResourceDisciplines//ResourceDisciplineDisciplineIdIds
                             where relation.ResourceId == resource.Id
                             select relation).Any())
                            .Select((discipline) => new DisciplineObject(discipline.Id, discipline.Url, discipline.DisplayNameDe, discipline.DisplayNameEn));

            if (resource.Visibility == null && resource.VisibilityId != null)
            {
                VisibilityModel visibilityModel = new VisibilityModel();
                resource.Visibility = visibilityModel.GetById(resource.VisibilityId.Value);
            }

            if (resource.License == null && resource.LicenseId != null)
            {
                LicenseModel licenseModel = new LicenseModel();
                resource.License = licenseModel.GetById(resource.LicenseId.Value);
            }

            ResourceTypeOptionObject resourceTypeOptionObject = null;
            if (resource.Type.DisplayName == "rds" && resource.ResourceTypeOptionId != null)
            {
                RDSResourceTypeModel rdsResourceTypeModel = new RDSResourceTypeModel();
                var rdsResourceType = rdsResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);
                resourceTypeOptionObject = new RDSResourceTypeObject(rdsResourceType.Id, rdsResourceType.BucketName, (int)rdsResourceType.Size);
            }
            else if (resource.Type.DisplayName == "s3" && resource.ResourceTypeOptionId != null)
            {
                S3ResourceTypeModel s3ResourceTypeModel = new S3ResourceTypeModel();
                var s3ResourceType = s3ResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);
                resourceTypeOptionObject = new S3ResourceTypeObject(s3ResourceType.Id, s3ResourceType.BucketName, null, null, s3ResourceType.ResourceUrl);
            }
            else if (resource.Type.DisplayName == "gitlab" && resource.ResourceTypeOptionId != null)
            {
                GitlabResourceTypeModel gitlabResourceTypeModel = new GitlabResourceTypeModel();
                var gitlabResourceType = gitlabResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);
                resourceTypeOptionObject = new GitlabResourceTypeObject(gitlabResourceType.Id, gitlabResourceType.RepositoryNumber, gitlabResourceType.RepositoryUrl, null);
            }
            else if (resource.Type.DisplayName == "linked" && resource.ResourceTypeOptionId != null)
            {
                LinkedResourceTypeModel linkedResourceTypeModel = new LinkedResourceTypeModel();
                var linkedResourceType = linkedResourceTypeModel.GetById(resource.ResourceTypeOptionId.Value);
                resourceTypeOptionObject = new LinkedResourceTypeObject(linkedResourceType.Id);
            }

            return new ResourceObject(
                resource.Id,
                resource.DisplayName,
                resource.ResourceName,
                resource.Description,
                resource.Keywords,
                resource.UsageRights,
                new ResourceTypeObject(resource.Type.Id, resource.Type.DisplayName),
                disciplines,
                (resource.Visibility != null) ? new VisibilityObject(resource.Visibility.Id, resource.Visibility.DisplayName) : null,
                (resource.License != null) ? new LicenseObject(resource.License.Id, resource.License.DisplayName) : null,
                JObject.FromObject(resourceTypeOptionObject),
                resource.ApplicationProfile,
                JToken.Parse(resource.FixedValues ?? "{}"),
                (resource.Creator != null) ? resource.Creator : null
            );
        }

        public override Expression<Func<Resource, Guid>> GetIdFromObject()
        {
            return databaseObject => databaseObject.Id;
        }

        public override Microsoft.EntityFrameworkCore.DbSet<Resource> GetITableFromDatabase(CoscineDB db)
        {
            return db.Resources;
        }

        public override void SetObjectId(Resource databaseObject, Guid id)
        {
            databaseObject.Id = id;
        }
    }
}
